<?php
require_once '../config/db.php';
header('Content-Type: application/json');

$customer_id = isset($_GET['customer_id']) ? intval($_GET['customer_id']) : 0;
if (!$customer_id) {
    echo json_encode(['success' => false, 'error' => 'Missing customer_id']);
    exit;
}

// Get detailed unpaid items with quantities, prices, and payment info including waivers
// Note: POS payments are included for informational purposes but only credit payments count toward debt reduction
$stmt = $pdo->prepare('
    SELECT 
        s.invoice_number,
        s.created_at as sale_date,
        ct.amount as credit_amount,
        (SELECT COALESCE(SUM(cp.amount), 0) FROM credit_payments cp WHERE cp.credit_transaction_id = ct.id AND (cp.is_waived = 0 OR cp.is_waived IS NULL) AND cp.payment_method != "refund") AS credit_paid_amount,
        (SELECT COALESCE(SUM(cp.amount), 0) FROM credit_payments cp WHERE cp.credit_transaction_id = ct.id AND cp.is_waived = 1) AS waived_amount,
        (SELECT COALESCE(SUM(cp.amount), 0) FROM credit_payments cp WHERE cp.credit_transaction_id = ct.id AND cp.payment_method = "refund") AS refunded_amount,
        (SELECT COALESCE(SUM(sp.amount), 0) FROM sale_payments sp WHERE sp.sale_id = s.id AND LOWER(sp.method) != "credit") AS pos_paid_amount,
        ct.id as credit_id,
        s.id as sale_id
    FROM credit_transactions ct
    JOIN sales s ON ct.sale_id = s.id
    WHERE ct.customer_id = ?
    HAVING (ct.amount - refunded_amount) - credit_paid_amount - waived_amount > 0
    ORDER BY s.created_at DESC
');
$stmt->execute([$customer_id]);

$items = [];
while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
    // Total paid should only include actual payments toward credit debt (not POS payments from original sales)
    $total_paid = $row['credit_paid_amount'];
    // Correct balance: (Credit Amount - Refunded Amount) - Credit Payments - Waived Amount
    // POS payments are already accounted for when creating the credit amount
    $refunded_amount = $row['refunded_amount'] ?? 0;
    $adjusted_credit_amount = $row['credit_amount'] - $refunded_amount;
    $balance = $adjusted_credit_amount - $row['credit_paid_amount'] - $row['waived_amount'];
    if ($balance > 0.001) {
        // Get all items for this invoice (excluding returned items) with serial numbers
        $stmt_items = $pdo->prepare('
            SELECT 
                p.name AS item_name, 
                si.quantity, 
                si.price,
                GROUP_CONCAT(DISTINCT psn.serial_number ORDER BY psn.created_at ASC SEPARATOR ", ") as serial_numbers
            FROM sale_items si
            JOIN products p ON si.product_id = p.id
            JOIN sales s ON si.sale_id = s.id
            LEFT JOIN product_serial_numbers psn ON si.product_id = psn.product_id 
                AND psn.status = "sold" 
                AND psn.source_id = si.sale_id
                AND psn.source_type IN ("credit_sale", "manual_adjustment")
            WHERE s.invoice_number = ?
            AND si.id NOT IN (
                SELECT DISTINCT ri.sale_item_id 
                FROM refund_items ri 
                JOIN refunds r ON ri.refund_id = r.id 
                WHERE r.sale_id = s.id
            )
            GROUP BY si.id, p.name, si.quantity, si.price
            ORDER BY p.name
        ');
        $stmt_items->execute([$row['invoice_number']]);
        $invoice_items = $stmt_items->fetchAll(PDO::FETCH_ASSOC);
        
        // Get payment methods used in POS
        $stmt_payments = $pdo->prepare('
            SELECT method, amount FROM sale_payments 
            WHERE sale_id = ? AND LOWER(method) != "credit"
            ORDER BY id
        ');
        $stmt_payments->execute([$row['sale_id']]);
        $pos_payments = $stmt_payments->fetchAll(PDO::FETCH_ASSOC);
        
        $items[] = [
            'invoice' => $row['invoice_number'],
            'invoice_number' => $row['invoice_number'],
            'sale_date' => $row['sale_date'],
            'credit_amount' => $adjusted_credit_amount,
            'original_credit_amount' => $row['credit_amount'],
            'amount' => $adjusted_credit_amount, // For compatibility
            'credit_paid_amount' => $row['credit_paid_amount'],
            'paid_amount' => $row['credit_paid_amount'], // For compatibility
            'waived_amount' => $row['waived_amount'],
            'refunded_amount' => $refunded_amount,
            'pos_paid_amount' => $row['pos_paid_amount'],
            'total_paid_amount' => $total_paid,
            'balance' => $balance,
            'items' => $invoice_items,
            'pos_payments' => $pos_payments
        ];
    }
}
echo json_encode(['success' => true, 'items' => $items]); 